/*******************************************************************************
 * Copyright (c) PLX Technology, Inc.
 *
 * PLX Technology Inc. licenses this source file under the GNU Lesser General Public
 * License (LGPL) version 2.  This source file may be modified or redistributed
 * under the terms of the LGPL and without express permission from PLX Technology.
 *
 * PLX Technology, Inc. provides this software AS IS, WITHOUT ANY WARRANTY,
 * EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  PLX makes no guarantee
 * or representations regarding the use of, or the results of the use of,
 * the software and documentation in terms of correctness, accuracy,
 * reliability, currentness, or otherwise; and you rely on the software,
 * documentation and results solely at your own risk.
 *
 * IN NO EVENT SHALL PLX BE LIABLE FOR ANY LOSS OF USE, LOSS OF BUSINESS,
 * LOSS OF PROFITS, INDIRECT, INCIDENTAL, SPECIAL OR CONSEQUENTIAL DAMAGES
 * OF ANY KIND.
 *
 ******************************************************************************/

/*******************************************************************************
 *
 * File Name:
 *
 *      ApiFunc.c
 *
 * Description:
 *
 *      Implements the PLX API functions
 *
 * Revision History:
 *
 *      02-01-13 : PLX SDK v7.00
 *
 ******************************************************************************/


#include "ApiFunc.h"
#include "Eep_6000.h"
#include "GlobalVars.h"
#include "PciFunc.h"
#include "PlxInterrupt.h"
#include "SuppFunc.h"




/*******************************************************************************
 *
 * Function   :  PlxDeviceFind
 *
 * Description:  Search for a specific device using device key parameters
 *
 ******************************************************************************/
PLX_STATUS
PlxDeviceFind(
    DEVICE_EXTENSION *pdx,
    PLX_DEVICE_KEY   *pKey,
    U16              *pDeviceNumber
    )
{
    U16            DeviceCount;
    BOOLEAN        bMatchId;
    BOOLEAN        bMatchLoc;
    DEVICE_OBJECT *fdo;


    DeviceCount = 0;

    // Get first device instance in list
    fdo = pdx->pDeviceObject->DriverObject->DeviceObject;

    // Compare with items in device list
    while (fdo != NULL)
    {
        // Get the device extension
        pdx = fdo->DeviceExtension;

        // Assume successful match
        bMatchLoc = TRUE;
        bMatchId  = TRUE;

        //
        // Compare device key information
        //

        // Compare Bus number
        if (pKey->bus != (U8)PCI_FIELD_IGNORE)
        {
            if (pKey->bus != pdx->Key.bus)
            {
                bMatchLoc = FALSE;
            }
        }

        // Compare Slot number
        if (pKey->slot != (U8)PCI_FIELD_IGNORE)
        {
            if (pKey->slot != pdx->Key.slot)
            {
                bMatchLoc = FALSE;
            }
        }

        // Compare Function number
        if (pKey->function != (U8)PCI_FIELD_IGNORE)
        {
            if (pKey->function != pdx->Key.function)
            {
                bMatchLoc = FALSE;
            }
        }

        //
        // Compare device ID information
        //

        // Compare Vendor ID
        if (pKey->VendorId != (U16)PCI_FIELD_IGNORE)
        {
            if (pKey->VendorId != pdx->Key.VendorId)
            {
                bMatchId = FALSE;
            }
        }

        // Compare Device ID
        if (pKey->DeviceId != (U16)PCI_FIELD_IGNORE)
        {
            if (pKey->DeviceId != pdx->Key.DeviceId)
            {
                bMatchId = FALSE;
            }
        }

        // Compare Subsystem Vendor ID
        if (pKey->SubVendorId != (U16)PCI_FIELD_IGNORE)
        {
            if (pKey->SubVendorId != pdx->Key.SubVendorId)
            {
                bMatchId = FALSE;
            }
        }

        // Compare Subsystem Device ID
        if (pKey->SubDeviceId != (U16)PCI_FIELD_IGNORE)
        {
            if (pKey->SubDeviceId != pdx->Key.SubDeviceId)
            {
                bMatchId = FALSE;
            }
        }

        // Compare Revision
        if (pKey->Revision != (U8)PCI_FIELD_IGNORE)
        {
            if (pKey->Revision != pdx->Key.Revision)
            {
                bMatchId = FALSE;
            }
        }

        // Check if match on location and ID
        if (bMatchLoc && bMatchId)
        {
            // Match found, check if it is the desired device
            if (DeviceCount == *pDeviceNumber)
            {
                // Copy the device information
                *pKey = pdx->Key;

                DebugPrintf((
                    "Criteria matched device %04X_%04X [b:%02x s:%02x f:%x]\n",
                    pdx->Key.DeviceId, pdx->Key.VendorId,
                    pdx->Key.bus, pdx->Key.slot, pdx->Key.function
                    ));

                return ApiSuccess;
            }

            // Increment device count
            DeviceCount++;
        }

        // Jump to next entry
        fdo = fdo->NextDevice;
    }

    // Return number of matched devices
    *pDeviceNumber = DeviceCount;

    DebugPrintf(("Criteria did not match any devices\n"));

    return ApiInvalidDeviceInfo;
}




/*******************************************************************************
 *
 * Function   :  PlxChipTypeGet
 *
 * Description:  Returns PLX chip type and revision
 *
 ******************************************************************************/
PLX_STATUS
PlxChipTypeGet(
    DEVICE_EXTENSION *pdx,
    U16              *pChipType,
    U8               *pRevision
    )
{
    *pChipType = pdx->Key.PlxChip;
    *pRevision = pdx->Key.PlxRevision;

    DebugPrintf((
        "PLX chip = %04X rev %02X\n",
        *pChipType, *pRevision
        ));

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxChipTypeSet
 *
 * Description:  Sets the PLX chip type dynamically
 *
 ******************************************************************************/
PLX_STATUS
PlxChipTypeSet(
    DEVICE_EXTENSION *pdx,
    U16               ChipType,
    U8                Revision
    )
{
    // Setting the PLX chip type is not supported in this PnP driver
    return ApiUnsupportedFunction;
}




/*******************************************************************************
 *
 * Function   :  PlxGetPortProperties
 *
 * Description:  Returns the current port information and status
 *
 ******************************************************************************/
PLX_STATUS
PlxGetPortProperties(
    DEVICE_EXTENSION *pdx,
    PLX_PORT_PROP    *pPortProp
    )
{
    DebugPrintf(("Device does not support PCI Express Capability\n"));

    // Set default properties
    RtlZeroMemory(pPortProp, sizeof(PLX_PORT_PROP));

    // Mark device as non-PCIe
    pPortProp->bNonPcieDevice = TRUE;

    // Default to a legacy endpoint
    pPortProp->PortType = PLX_PORT_LEGACY_ENDPOINT;

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxPciDeviceReset
 *
 * Description:  Resets a device using software reset feature of PLX chip
 *
 ******************************************************************************/
PLX_STATUS
PlxPciDeviceReset(
    DEVICE_EXTENSION *pdx
    )
{
    // Device reset not implemented
    return ApiUnsupportedFunction;
}




/*******************************************************************************
 *
 * Function   :  PlxRegisterRead
 *
 * Description:  Reads a PLX-specific register
 *
 ******************************************************************************/
U32
PlxRegisterRead(
    DEVICE_EXTENSION *pdx,
    U32               offset,
    PLX_STATUS       *pStatus,
    BOOLEAN           bAdjustForPort
    )
{
    // 6000 series devices do not have PLX-specific local registers
    *pStatus = ApiUnsupportedFunction;
    return 0;
}




/*******************************************************************************
 *
 * Function   :  PlxRegisterWrite
 *
 * Description:  Writes to a PLX-specific register
 *
 ******************************************************************************/
PLX_STATUS
PlxRegisterWrite(
    DEVICE_EXTENSION *pdx,
    U32               offset,
    U32               value,
    BOOLEAN           bAdjustForPort
    )
{
    // 6000 series devices do not have PLX-specific local registers
    return ApiUnsupportedFunction;
}




/*******************************************************************************
 *
 * Function   :  PlxPciBarProperties
 *
 * Description:  Returns the properties of a PCI BAR space
 *
 ******************************************************************************/
PLX_STATUS
PlxPciBarProperties(
    DEVICE_EXTENSION *pdx,
    U8                BarIndex,
    PLX_PCI_BAR_PROP *pBarProperties
    )
{
    // Verify BAR number
    switch (BarIndex)
    {
        case 0:
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
            break;

        default:
            return ApiInvalidIndex;
    }

    // Return BAR properties
    *pBarProperties = pdx->PciBar[BarIndex].Properties;

    // Display BAR properties if enabled
    if (pdx->PciBar[BarIndex].Properties.Size == 0)
    {
        DebugPrintf(("BAR %d is disabled\n", BarIndex));
        return ApiSuccess;
    }

    DebugPrintf((
        "    PCI BAR %d: %08I64X\n",
        BarIndex, pdx->PciBar[BarIndex].Properties.BarValue
        ));

    DebugPrintf((
        "    Phys Addr: %08I64X\n",
        pdx->PciBar[BarIndex].Properties.Physical
        ));

    DebugPrintf((
        "    Size     : %08I64X (%I64d %s)\n",
        pdx->PciBar[BarIndex].Properties.Size,
        pdx->PciBar[BarIndex].Properties.Size < ((U64)1 << 10) ?
            pdx->PciBar[BarIndex].Properties.Size :
            pdx->PciBar[BarIndex].Properties.Size >> 10,
        pdx->PciBar[BarIndex].Properties.Size < ((U64)1 << 10) ? "Bytes" : "KB"
        ));

    DebugPrintf((
        "    Property : %sPrefetchable %d-bit\n",
        (pdx->PciBar[BarIndex].Properties.Flags & PLX_BAR_FLAG_PREFETCHABLE) ? "" : "Non-",
        (pdx->PciBar[BarIndex].Properties.Flags & PLX_BAR_FLAG_64_BIT) ? 64 : 32
        ));

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxPciBarMap
 *
 * Description:  Map a PCI BAR Space into User virtual space
 *
 ******************************************************************************/
PLX_STATUS
PlxPciBarMap(
    DEVICE_EXTENSION *pdx,
    U8                BarIndex,
    VOID             *pUserVa,
    VOID             *pOwner
    )
{
    U8                   i;
    PMDL                 pMdl[MAX_VIRTUAL_ADDR];
    VOID                *Va[MAX_VIRTUAL_ADDR];
    VOID                *VaBase;
    BOOLEAN              bMapError;
    PLX_USER_MAPPING    *pMapObject;
    MEMORY_CACHING_TYPE  CacheMode;


    // Set default address
    *(PLX_UINT_PTR*)pUserVa = 0;

    // Verify BAR is of type memory
    if (pdx->PciBar[BarIndex].Properties.Flags & PLX_BAR_FLAG_IO)
    {
        DebugPrintf((
            "ERROR - BAR %d is an I/O space, cannot map to user space\n",
            BarIndex
            ));

        return ApiInvalidPciSpace;
    }

    // Check if the space is valid
    if (pdx->PciBar[BarIndex].Properties.Physical == 0)
    {
        DebugPrintf((
            "ERROR - BAR %d address (%08lx) is invalid\n",
            BarIndex, (PLX_UINT_PTR)pdx->PciBar[BarIndex].Properties.Physical
            ));

        return ApiInvalidAddress;
    }

    // Verify Kernel virtual address
    if (pdx->PciBar[BarIndex].pVa == NULL)
    {
        DebugPrintf((
            "ERROR - Kernel Virtual address for BAR %d is not valid\n",
            BarIndex
            ));

        return ApiInvalidAddress;
    }

    // Make sure memory size is valid
    if (pdx->PciBar[BarIndex].Properties.Size == 0)
    {
        DebugPrintf((
            "ERROR - Size of BAR %d is 0\n",
            BarIndex
            ));

        return ApiInvalidSize;
    }

    // Verify the MDL
    if (pdx->PciBar[BarIndex].pMdl == NULL)
    {
        DebugPrintf((
            "ERROR - MDL does not exist for BAR %d\n",
            BarIndex
            ));

        return ApiInsufficientResources;
    }

    // Make sure there is enough space to store all virtual addresses
    if (pdx->PciBar[BarIndex].Properties.Size > (MAX_VIRTUAL_ADDR * MAX_MDL_SIZE))
    {
        DebugPrintf((
            "ERROR - Not enough virtual addresses for space of size %ldMB\n",
            ((PLX_UINT_PTR)pdx->PciBar[BarIndex].Properties.Size >> 20)
            ));

        return ApiInsufficientResources;
    }

    // Set initial values
    i         = 0;
    pMdl[0]   = pdx->PciBar[BarIndex].pMdl;
    VaBase    = NULL;
    bMapError = FALSE;

    /***********************************************************
     * For pre-fetchable region, enable write-combining
     * Cached mode seems to have no effect, but write-combining
     * results in much better CPU write performance.
     **********************************************************/
    if (pdx->PciBar[BarIndex].Properties.Flags & PLX_BAR_FLAG_PREFETCHABLE)
        CacheMode = MmWriteCombined;
    else
        CacheMode = MmNonCached;

    // Clear the virtual addresses
    RtlZeroMemory(
        Va,
        MAX_VIRTUAL_ADDR * sizeof(VOID*)
        );

    // Attempt to map the BAR into user space
    while ((i < MAX_VIRTUAL_ADDR) && (pMdl[i] != NULL))
    {
        try
        {
            // Attempt to map next region
            Va[i] =
                MmMapLockedPagesSpecifyCache(
                    pMdl[i],             // MDL for region
                    UserMode,            // User or kernel mode?
                    CacheMode,           // Cache type?
                    VaBase,              // User address to use
                    FALSE,               // Do not issue a bug check (KernelMode only)
                    NormalPagePriority   // Priority of success
                    );
        }
        except (EXCEPTION_EXECUTE_HANDLER)
        {
            DebugPrintf(("EXCEPTION: Raised by MmMapLockedPagesSpecifyCache() for MDL #%d\n", i));
            Va[i] = NULL;
        }

        // Check if the mapping succeeded
        if (Va[i] == NULL)
        {
            DebugPrintf((
                "ERROR - Unable to map PCI BAR %d (%08lx) ==> User space\n",
                BarIndex, (PLX_UINT_PTR)pdx->PciBar[BarIndex].Properties.Physical
                ));

            bMapError = TRUE;
        }
        else if (VaBase != NULL)
        {
            // Make sure VA address is consecutive
            if (VaBase != Va[i])
            {
                DebugPrintf((
                    "ERROR - Unable to assign contiguous virtual addresses for PCI BAR %d (%08lx) ==> User space\n",
                    BarIndex, (PLX_UINT_PTR)pdx->PciBar[BarIndex].Properties.Physical
                    ));

                bMapError = TRUE;
            }
        }

        if (bMapError)
        {
            // Release any previous mappings
            while (i != 0)
            {
                i--;

                MmUnmapLockedPages(
                    Va[i],
                    pMdl[i]
                    );
            }

            return ApiInsufficientResources;
        }

        // Adjust to next consecutive base address
        VaBase = ((U8*)Va[i]) + MmGetMdlByteCount( pMdl[i] );

        // Go to next entry
        i++;
        pMdl[i] = pMdl[i-1]->Next;
    }

    // Return initial base address
    *(PLX_UINT_PTR*)pUserVa = (PLX_UINT_PTR)Va[0];

    DebugPrintf((
        "Mapped PCI BAR %d (%08lx) ==> User VA (%08lx) (owner=%p)\n",
        BarIndex, (PLX_UINT_PTR)pdx->PciBar[BarIndex].Properties.Physical,
        *(PLX_UINT_PTR*)pUserVa, pOwner
        ));

    // Add mapping to list for later unmapping
    pMapObject =
        ExAllocatePoolWithTag(
            NonPagedPool,
            sizeof(PLX_USER_MAPPING),
            '_XLP'        // Tag = "PLX_" in reverse order
            );

    if (pMapObject == NULL)
    {
        DebugPrintf(("ERROR - Unable to save mapping, insufficient memory\n"));
    }
    else
    {
        // Record mapping properties
        pMapObject->pOwner   = pOwner;
        pMapObject->BarIndex = BarIndex;

        // Record the virtual address
        RtlCopyMemory(
            pMapObject->pUserVa,
            Va,
            MAX_VIRTUAL_ADDR * sizeof(VOID*)
            );

        // Save the mapping for later cleanup
        ExInterlockedInsertTailList(
            &(pdx->List_BarMappings),
            &(pMapObject->ListEntry),
            &(pdx->Lock_BarMappingsList)
            );
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxPciBarUnmap
 *
 * Description:  Unmap a previously mapped PCI BAR Space from User virtual space
 *
 ******************************************************************************/
PLX_STATUS
PlxPciBarUnmap(
    DEVICE_EXTENSION *pdx,
    VOID             *UserVa,
    VOID             *pOwner
    )
{
    U8                i;
    PMDL              pMdl;
    KIRQL             IrqL_Original;
    PLIST_ENTRY       pEntry;
    PLX_USER_MAPPING *pMapObject;


    // Find the mapped object
    KeAcquireSpinLock(
        &(pdx->Lock_BarMappingsList),
        &IrqL_Original
        );

    pEntry = pdx->List_BarMappings.Flink;

    while (pEntry != &(pdx->List_BarMappings))
    {
        pMapObject =
            CONTAINING_RECORD(
                pEntry,
                PLX_USER_MAPPING,
                ListEntry
                );

        if ((pMapObject->pOwner == pOwner) &&
            (pMapObject->pUserVa[0] == UserVa))
        {
            // Remove map object from list
            RemoveEntryList(
                pEntry
                );

            KeReleaseSpinLock(
                &(pdx->Lock_BarMappingsList),
                IrqL_Original
                );

            DebugPrintf((
                "Unmap User VA (%p) for BAR %d (owner=%p)\n",
                pMapObject->pUserVa[0], pMapObject->BarIndex, pOwner
                ));

            // Unmap the space
            i    = 0;
            pMdl = pdx->PciBar[pMapObject->BarIndex].pMdl;

            while ((i < MAX_VIRTUAL_ADDR) && (pMapObject->pUserVa[i] != NULL))
            {
                MmUnmapLockedPages(
                    pMapObject->pUserVa[i],
                    pMdl
                    );

                i++;
                pMdl = pMdl->Next;
            }

            // Release the list object
            ExFreePool(
                pMapObject
                );

            return ApiSuccess;
        }

        // Jump to next item
        pEntry = pEntry->Flink;
    }

    KeReleaseSpinLock(
        &(pdx->Lock_BarMappingsList),
        IrqL_Original
        );

    DebugPrintf((
        "ERROR - Map object not found in list...\n"
        ));

    return ApiInvalidAddress;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromPresent
 *
 * Description:  Returns the state of the EEPROM as reported by the PLX device
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromPresent(
    DEVICE_EXTENSION *pdx,
    U8               *pStatus
    )
{
    BOOLEAN bFlag;


    /**************************************************************
     * Since PCI 6000 devices do not report whether an EEPROM
     * is present or not, the driver manually probes for it.
     *************************************************************/

    // Probe for EEPROM
    PlxEepromProbe(
        pdx,
        &bFlag
        );

    if (bFlag)
        *pStatus = PLX_EEPROM_STATUS_VALID;
    else
        *pStatus = PLX_EEPROM_STATUS_NONE;

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromProbe
 *
 * Description:  Probes for the presence of an EEPROM
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromProbe(
    DEVICE_EXTENSION *pdx,
    U8               *pFlag
    )
{
    U16        OffsetProbe;
    U16        ValueRead;
    U16        ValueWrite;
    U16        ValueOriginal;
    PLX_STATUS status;


    // Default to no EEPROM present
    *pFlag = FALSE;

    // Set EEPROM offset to use for probe
    OffsetProbe = 0x50;     // Use area outside of PLX portion

    DebugPrintf(("Probe EEPROM at offset %02xh\n", OffsetProbe));

    // Get the current value
    status =
        PlxEepromReadByOffset_16(
            pdx,
            OffsetProbe,
            &ValueOriginal
            );

    if (status != ApiSuccess)
        return status;

    // Prepare inverse value to write
    ValueWrite = (U16)~ValueOriginal;

    // Write inverse of original value
    status =
        PlxEepromWriteByOffset_16(
            pdx,
            OffsetProbe,
            ValueWrite
            );

    if (status != ApiSuccess)
        return status;

    // Read updated value
    status =
        PlxEepromReadByOffset_16(
            pdx,
            OffsetProbe,
            &ValueRead
            );

    if (status != ApiSuccess)
        return status;

    // Check if value was written properly
    if (ValueRead == ValueWrite)
    {
        DebugPrintf(("Probe detected an EEPROM present\n"));

        *pFlag = TRUE;

        // Restore the original value
        PlxEepromWriteByOffset_16(
            pdx,
            OffsetProbe,
            ValueOriginal
            );
    }
    else
    {
        DebugPrintf(("Probe did not detect an EEPROM\n"));
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromCrcGet
 *
 * Description:  Returns the EEPROM CRC and its status
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromCrcGet(
    DEVICE_EXTENSION *pdx,
    U32              *pCrc,
    U8               *pCrcStatus
    )
{
    // Clear return value
    *pCrc       = 0;
    *pCrcStatus = PLX_CRC_UNSUPPORTED;

    // CRC not supported
    return ApiUnsupportedFunction;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromCrcUpdate
 *
 * Description:  Updates the EEPROM CRC
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromCrcUpdate(
    DEVICE_EXTENSION *pdx,
    U32              *pCrc,
    BOOLEAN           bUpdateEeprom
    )
{
    // Clear return value
    *pCrc = 0;

    // CRC not supported
    return ApiUnsupportedFunction;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromReadByOffset
 *
 * Description:  Read a 32-bit value from the EEPROM at a specified offset
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromReadByOffset(
    DEVICE_EXTENSION *pdx,
    U32               offset,
    U32              *pValue
    )
{
    // Make sure offset is aligned on 32-bit boundary
    if (offset & (3 << 0))
    {
        *pValue = 0;
        return ApiInvalidOffset;
    }

    Plx6000_EepromReadByOffset_16(
        pdx,
        offset,
        (U16*)pValue
        );

    Plx6000_EepromReadByOffset_16(
        pdx,
        offset + sizeof(U16),
        (U16*)((U8*)pValue + sizeof(U16))
        );

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromWriteByOffset
 *
 * Description:  Write a 32-bit value to the EEPROM at a specified offset
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromWriteByOffset(
    DEVICE_EXTENSION *pdx,
    U32               offset,
    U32               value
    )
{
    // Make sure offset is aligned on 32-bit boundary
    if (offset & (3 << 0))
        return ApiInvalidOffset;

    Plx6000_EepromWriteByOffset_16(
        pdx,
        offset,
        (U16)value
        );

    Plx6000_EepromWriteByOffset_16(
        pdx,
        offset + sizeof(U16),
        (U16)(value >> 16)
        );

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxEepromReadByOffset_16
 *
 * Description:  Read a 16-bit value from the EEPROM at a specified offset
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromReadByOffset_16(
    DEVICE_EXTENSION *pdx,
    U32               offset,
    U16              *pValue
    )
{
    // Make sure offset is aligned on 16-bit boundary
    if (offset & (1 << 0))
    {
        *pValue = 0;
        return ApiInvalidOffset;
    }

    return Plx6000_EepromReadByOffset_16(
        pdx,
        offset,
        pValue
        );
}




/*******************************************************************************
 *
 * Function   :  PlxEepromWriteByOffset_16
 *
 * Description:  Write a 16-bit value to the EEPROM at a specified offset
 *
 ******************************************************************************/
PLX_STATUS
PlxEepromWriteByOffset_16(
    DEVICE_EXTENSION *pdx,
    U32               offset,
    U16               value
    )
{
    // Make sure offset is aligned on 16-bit boundary
    if (offset & (1 << 0))
        return ApiInvalidOffset;

    return Plx6000_EepromWriteByOffset_16(
        pdx,
        offset,
        value
        );
}




/*******************************************************************************
 *
 * Function   :  PlxPciIoPortTransfer
 *
 * Description:  Read or Write from/to an I/O port
 *
 ******************************************************************************/
PLX_STATUS
PlxPciIoPortTransfer(
    U64              IoPort,
    VOID            *pBuffer,
    U32              SizeInBytes,
    PLX_ACCESS_TYPE  AccessType,
    BOOLEAN          bReadOperation
    )
{
    U8 AccessSize;


    if (pBuffer == NULL)
        return ApiNullParam;

    // Verify size & type
    switch (AccessType)
    {
        case BitSize8:
            AccessSize = sizeof(U8);
            break;

        case BitSize16:
            if (IoPort & (1 << 0))
            {
                DebugPrintf(("ERROR - I/O port not aligned on 16-bit boundary\n"));
                return ApiInvalidAddress;
            }

            if (SizeInBytes & (1 << 0))
            {
                DebugPrintf(("ERROR - Byte count not aligned on 16-bit boundary\n"));
                return ApiInvalidSize;
            }
            AccessSize = sizeof(U16);
            break;

        case BitSize32:
            if (IoPort & 0x3)
            {
                DebugPrintf(("ERROR - I/O port not aligned on 32-bit boundary\n"));
                return ApiInvalidAddress;
            }

            if (SizeInBytes & 0x3)
            {
                DebugPrintf(("ERROR - Byte count not aligned on 32-bit boundary\n"));
                return ApiInvalidSize;
            }
            AccessSize = sizeof(U32);
            break;

        default:
            return ApiInvalidAccessType;
    }

    while (SizeInBytes)
    {
        if (bReadOperation)
        {
            switch (AccessType)
            {
                case BitSize8:
                    *(U8*)pBuffer = IO_PORT_READ_8( IoPort );
                    break;

                case BitSize16:
                    *(U16*)pBuffer = IO_PORT_READ_16( IoPort );
                    break;

                case BitSize32:
                    *(U32*)pBuffer = IO_PORT_READ_32( IoPort );
                    break;

                default:
                    // Added to avoid compiler warnings
                    break;
            }
        }
        else
        {
            switch (AccessType)
            {
                case BitSize8:
                    IO_PORT_WRITE_8(
                        IoPort,
                        *(U8*)pBuffer
                        );
                    break;

                case BitSize16:
                    IO_PORT_WRITE_16(
                        IoPort,
                        *(U16*)pBuffer
                        );
                    break;

                case BitSize32:
                    IO_PORT_WRITE_32(
                        IoPort,
                        *(U32*)pBuffer
                        );
                    break;

                default:
                    // Added to avoid compiler warnings
                    break;
            }
        }

        // Adjust pointer & byte count
        pBuffer      = (VOID*)((PLX_UINT_PTR)pBuffer + AccessSize);
        SizeInBytes -= AccessSize;
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxPciPhysicalMemoryAllocate
 *
 * Description:  Allocate physically contiguous page-locked memory
 *
 ******************************************************************************/
PLX_STATUS
PlxPciPhysicalMemoryAllocate(
    DEVICE_EXTENSION *pdx,
    PLX_PHYSICAL_MEM *pPciMem,
    BOOLEAN           bSmallerOk,
    VOID             *pOwner
    )
{
    U32                  SizeToMap;
    U32                  SizeRemain;
    U32                  DecrementAmount;
    VOID                *pVa;
    PMDL                 pMdl;
    PMDL                 pMdl_Previous;
    PLX_PHYS_MEM_OBJECT *pMemObject;


    // Initialize buffer information
    pPciMem->UserAddr     = 0;
    pPciMem->PhysicalAddr = 0;
    pPciMem->CpuPhysical  = 0;

    /*******************************************************
     * Verify size
     *
     * A size of 0 is valid because this function may
     * be called to allocate a common buffer of size 0;
     * therefore, the information is reset & return sucess.
     ******************************************************/
    if (pPciMem->Size == 0)
    {
        return ApiSuccess;
    }

    // Allocate memory for new list object
    pMemObject =
        ExAllocatePoolWithTag(
            NonPagedPool,
            sizeof(PLX_PHYS_MEM_OBJECT),
            '_XLP'        // Tag = "PLX_" in reverse order
            );

    if (pMemObject == NULL)
    {
        DebugPrintf(("ERROR - Memory allocation for list object failed\n"));
        return ApiInsufficientResources;
    }

    // Clear object
    RtlZeroMemory( pMemObject, sizeof(PLX_PHYS_MEM_OBJECT) );

    // Set buffer request size
    pMemObject->Size = pPciMem->Size;

    // Set Cacheability of the buffer. System RAM is always assumed as cacheable
    pMemObject->bCacheable = MmCached;

    // Setup amount to reduce on failure
    DecrementAmount = (pPciMem->Size / 10);

    DebugPrintf((
        "Attempt to allocate physical memory (%d Kb)...\n",
        (pPciMem->Size >> 10)
        ));

    do
    {
        // Attempt to allocate the buffer
        pMemObject->pKernelVa =
            Plx_dma_buffer_alloc(
                pdx,
                pMemObject
                );

        if (pMemObject->pKernelVa == NULL)
        {
            // Reduce memory request size if requested
            if (bSmallerOk && (pMemObject->Size > PAGE_SIZE))
            {
                pMemObject->Size -= DecrementAmount;
            }
            else
            {
                // Release the list object
                ExFreePool(
                    pMemObject
                    );

                DebugPrintf(("ERROR - Physical memory allocation failed\n"));

                pPciMem->Size = 0;

                return ApiInsufficientResources;
            }
        }
    }
    while (pMemObject->pKernelVa == NULL);

    // Record buffer owner
    pMemObject->pOwner = pOwner;

    /******************************************************************
     * A future mapping into user space will require an MDL. Due to
     * the limited region a single MDL can describe, multiple MDL's
     * must be allocated and chained together via the 'Next' field.
     *****************************************************************/

    // Set initial values
    pVa        = pMemObject->pKernelVa;
    SizeRemain = (U32)pMemObject->Size;

    while (SizeRemain != 0)
    {
        // Determine size for MDL
        if (SizeRemain <= MAX_MDL_SIZE)
        {
            SizeToMap = SizeRemain;
        }
        else
        {
            SizeToMap = MAX_MDL_SIZE;
        }

        // Get an MDL for future mapping into user space
        pMdl =
            IoAllocateMdl(
                pVa,
                SizeToMap,
                FALSE,          // Is this a secondary buffer?
                FALSE,          // Charge quota?
                NULL            // No IRP associated with MDL
                );

        // Check if the MDL allocation succeeded
        if (pMdl == NULL)
        {
            DebugPrintf(("ERROR - MDL allocation for space failed\n"));

            // Release any created MDLs
            while (pMemObject->pMdl != NULL)
            {
                pMdl = pMemObject->pMdl;
                pMemObject->pMdl = pMdl->Next;
                IoFreeMdl( pMdl );
            }

            return STATUS_SUCCESS;
        }

        // Build the MDL
        MmBuildMdlForNonPagedPool( pMdl );

        // Clear next MDL pointer
        pMdl->Next = NULL;

        // Store MDL
        if (pMemObject->pMdl == NULL)
        {
            // Insert first MDL
            pMemObject->pMdl = pMdl;
        }
        else
        {
            // Add new MDL to list
            pMdl_Previous->Next = pMdl;
        }

        // Update previous pointer
        pMdl_Previous = pMdl;

        // Adjust for next virtual address
        (U8*)pVa += SizeToMap;

        // Adjust remaining count
        SizeRemain -= SizeToMap;
    }

    // Initialize mappings list
    InitializeListHead(
        &(pMemObject->List_Mappings)
        );

    KeInitializeSpinLock(
        &(pMemObject->Lock_MappingsList)
        );

    // Return buffer information
    pPciMem->Size         = pMemObject->Size;
    pPciMem->PhysicalAddr = pMemObject->BusPhysical;
    pPciMem->CpuPhysical  = pMemObject->CpuPhysical;

    // Add buffer object to list
    ExInterlockedInsertTailList(
        &(pdx->List_PhysicalMem),
        &(pMemObject->ListEntry),
        &(pdx->Lock_PhysicalMemList)
        );

    // Check if this is the common buffer allocation
    if (pdx == pOwner)
    {
        // Store common buffer information
        pGbl_CommonBuffer = pMemObject;
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxPciPhysicalMemoryFree
 *
 * Description:  Free previously allocated physically contiguous page-locked memory
 *
 ******************************************************************************/
PLX_STATUS
PlxPciPhysicalMemoryFree(
    DEVICE_EXTENSION *pdx,
    PLX_PHYSICAL_MEM *pPciMem
    )
{
    PMDL                 pMdl;
    KIRQL                IrqL_Original;
    PLIST_ENTRY          pEntry;
    PLX_PHYS_MEM_OBJECT *pMemObject;


    KeAcquireSpinLock(
        &(pdx->Lock_PhysicalMemList),
        &IrqL_Original
        );

    pEntry = pdx->List_PhysicalMem.Flink;

    // Traverse list to find the desired list object
    while (pEntry != &(pdx->List_PhysicalMem))
    {
        // Get the object
        pMemObject =
            CONTAINING_RECORD(
                pEntry,
                PLX_PHYS_MEM_OBJECT,
                ListEntry
                );

        // Check if the physical addresses matches
        if (pMemObject->BusPhysical == pPciMem->PhysicalAddr)
        {
            // Release lock until mappings are removed
            KeReleaseSpinLock(
                &(pdx->Lock_PhysicalMemList),
                IrqL_Original
                );

            // Make sure all mappings to this memory are unmapped
            PlxPciPhysicalMemoryUnmapAll_ByOwner(
                pdx,
                pMemObject,
                NULL         // Ignore owner to remove all mappings
                );

            // Remove the object from the list
            KeAcquireSpinLock(
                &(pdx->Lock_PhysicalMemList),
                &IrqL_Original
                );

            RemoveEntryList(
                pEntry
                );

            KeReleaseSpinLock(
                &(pdx->Lock_PhysicalMemList),
                IrqL_Original
                );

            // Release the MDLs describing the region
            while (pMemObject->pMdl != NULL)
            {
                pMdl = pMemObject->pMdl;
                pMemObject->pMdl = pMdl->Next;
                IoFreeMdl( pMdl );
            }

            // Release the buffer
            Plx_dma_buffer_free(
                pdx,
                pMemObject
                );

            // Release the list object
            ExFreePool( pMemObject );

            // Check if this is the common buffer release
            if (pGbl_CommonBuffer == pMemObject)
            {
                // Clear common buffer information
                pGbl_CommonBuffer = NULL;
            }

            return ApiSuccess;
        }

        // Jump to next item in the list
        pEntry = pEntry->Flink;
    }

    KeReleaseSpinLock(
        &(pdx->Lock_PhysicalMemList),
        IrqL_Original
        );

    DebugPrintf(("ERROR - buffer object not found in list\n"));

    return ApiInvalidData;
}




/*******************************************************************************
 *
 * Function   :  PlxPciPhysicalMemoryMap
 *
 * Description:  Maps physical memory to User virtual address space
 *
 ******************************************************************************/
PLX_STATUS
PlxPciPhysicalMemoryMap(
    DEVICE_EXTENSION *pdx,
    PLX_PHYSICAL_MEM *pPciMem,
    VOID             *pOwner
    )
{
    U8                   i;
    PMDL                 pMdl[MAX_VIRTUAL_ADDR];
    VOID                *Va[MAX_VIRTUAL_ADDR];
    VOID                *VaBase;
    BOOLEAN              bMapError;
    KIRQL                IrqL_Original;
    BOOLEAN              bFound;
    PLIST_ENTRY          pEntry;
    PLX_USER_MAPPING    *pMapObject;
    PLX_PHYS_MEM_OBJECT *pMemObject;


    // Set default return value
    pPciMem->UserAddr = 0;

    pMemObject = NULL;

    // Check if memory object is common buffer
    if (pGbl_CommonBuffer != NULL)
    {
        if (pPciMem->PhysicalAddr == pGbl_CommonBuffer->BusPhysical)
        {
            pMemObject = pGbl_CommonBuffer;
        }
    }

    // Find the memory object to map
    if (pMemObject == NULL)
    {
        // Find the object in the list
        KeAcquireSpinLock(
            &(pdx->Lock_PhysicalMemList),
            &IrqL_Original
            );

        pEntry = pdx->List_PhysicalMem.Flink;

        bFound = FALSE;

        // Traverse list to find the desired list object
        while (!bFound && (pEntry != &(pdx->List_PhysicalMem)))
        {
            // Get the object
            pMemObject =
                CONTAINING_RECORD(
                    pEntry,
                    PLX_PHYS_MEM_OBJECT,
                    ListEntry
                    );

            // Check if the physical addresses matches
            if (pMemObject->BusPhysical == pPciMem->PhysicalAddr)
            {
                bFound = TRUE;
            }
            else
            {
                // Jump to next item in the list
                pEntry = pEntry->Flink;
            }
        }

        KeReleaseSpinLock(
            &(pdx->Lock_PhysicalMemList),
            IrqL_Original
            );

        if (!bFound)
        {
            DebugPrintf(("ERROR - Physical memory object not found in list, unable to map\n"));
            return ApiInvalidAddress;
        }
    }

    // Verify an MDL for the memory
    if (pMemObject->pMdl == NULL)
    {
        DebugPrintf(("ERROR - MDL does not exist for this memory, cannot map to user space\n"));
        return ApiInsufficientResources;
    }

    // Make sure there is enough space to store all virtual addresses
    if (pMemObject->Size > (MAX_VIRTUAL_ADDR * MAX_MDL_SIZE))
    {
        DebugPrintf((
            "ERROR - Not enough virtual addresses for memory of size %ldMB\n",
            (pMemObject->Size >> 20)
            ));

        return ApiInsufficientResources;
    }

    // Set initial values
    i         = 0;
    pMdl[0]   = pMemObject->pMdl;
    VaBase    = NULL;
    bMapError = FALSE;

    // Clear the virtual addresses
    RtlZeroMemory( Va, MAX_VIRTUAL_ADDR * sizeof(VOID*) );

    // Attempt to map into user space
    while ((i < MAX_VIRTUAL_ADDR) && (pMdl[i] != NULL))
    {
        try
        {
            // Attempt to map next region
            Va[i] =
                MmMapLockedPagesSpecifyCache(
                    pMdl[i],             // MDL for region
                    UserMode,            // User or kernel mode?
                    MmCached,            // System RAM is always Cached (otherwise mapping fails)
                    VaBase,              // User address to use
                    FALSE,               // Do not issue a bug check (KernelMode only)
                    HighPagePriority     // Priority of success
                    );
        }
        except (EXCEPTION_EXECUTE_HANDLER)
        {
            DebugPrintf(("EXCEPTION: Raised by MmMapLockedPagesSpecifyCache() for MDL #%d\n", i));
            Va[i] = NULL;
        }

        // Check if the mapping succeeded
        if (Va[i] == NULL)
        {
            DebugPrintf((
                "ERROR - Unable to map Physical address (%08lx) ==> User Space\n",
                (PLX_UINT_PTR)pMemObject->CpuPhysical
                ));

            bMapError = TRUE;
        }
        else if (VaBase != NULL)
        {
            // Make sure VA address is consecutive
            if (VaBase != Va[i])
            {
                DebugPrintf((
                    "ERROR - Unable to assign contiguous virtual addresses for memory (%08lx) ==> User space\n",
                    (PLX_UINT_PTR)pMemObject->CpuPhysical
                    ));

                bMapError = TRUE;
            }
        }

        if (bMapError)
        {
            // Release any previous mappings
            while (i != 0)
            {
                i--;

                MmUnmapLockedPages(
                    Va[i],
                    pMdl[i]
                    );
            }

            return ApiInsufficientResources;
        }

        // Adjust to next consecutive base address
        VaBase = ((U8*)Va[i]) + MmGetMdlByteCount( pMdl[i] );

        // Go to next entry
        i++;
        pMdl[i] = pMdl[i-1]->Next;
    }

    DebugPrintf((
        "Mapped Physical (%08lx) ==> VA (%p) (owner=%p)\n",
        (PLX_UINT_PTR)pMemObject->CpuPhysical, Va[0], pOwner
        ));

    // Return virtual address
    pPciMem->UserAddr = (PLX_UINT_PTR)Va[0];

    // Save mapping to list for this physical memory
    pMapObject =
        ExAllocatePoolWithTag(
            NonPagedPool,
            sizeof(PLX_USER_MAPPING),
            '_XLP'        // Tag = "PLX_" in reverse order
            );

    if (pMapObject == NULL)
    {
        DebugPrintf(("ERROR - Unable to save mapping, insufficient memory\n"));
    }
    else
    {
        // Record the owner
        pMapObject->pOwner = pOwner;

        // Record the virtual address
        RtlCopyMemory(
            pMapObject->pUserVa,
            Va,
            MAX_VIRTUAL_ADDR * sizeof(VOID*)
            );

        // Save the mapping for later cleanup
        ExInterlockedInsertTailList(
            &(pMemObject->List_Mappings),
            &(pMapObject->ListEntry),
            &(pMemObject->Lock_MappingsList)
            );
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxPciPhysicalMemoryUnmap
 *
 * Description:  Unmap physical memory from User virtual address space
 *
 ******************************************************************************/
PLX_STATUS
PlxPciPhysicalMemoryUnmap(
    DEVICE_EXTENSION *pdx,
    PLX_PHYSICAL_MEM *pPciMem,
    VOID             *pOwner
    )
{
    U8                   i;
    PMDL                 pMdl;
    VOID                *pUserVa;
    KIRQL                IrqL_Original;
    BOOLEAN              bFound;
    PLIST_ENTRY          pEntry;
    PLX_USER_MAPPING    *pMapObject;
    PLX_PHYS_MEM_OBJECT *pMemObject;


    pMemObject = NULL;

    // Check if memory object is common buffer
    if (pGbl_CommonBuffer != NULL)
    {
        if (pPciMem->PhysicalAddr == pGbl_CommonBuffer->BusPhysical)
        {
            pMemObject = pGbl_CommonBuffer;
        }
    }

    // Find the memory object to unmap
    if (pMemObject == NULL)
    {
        // Find the object in the list
        KeAcquireSpinLock(
            &(pdx->Lock_PhysicalMemList),
            &IrqL_Original
            );

        pEntry = pdx->List_PhysicalMem.Flink;

        bFound = FALSE;

        // Traverse list to find the desired list object
        while (!bFound && (pEntry != &(pdx->List_PhysicalMem)))
        {
            // Get the object
            pMemObject =
                CONTAINING_RECORD(
                    pEntry,
                    PLX_PHYS_MEM_OBJECT,
                    ListEntry
                    );

            // Check if the physical addresses matches
            if (pMemObject->BusPhysical == pPciMem->PhysicalAddr)
            {
                bFound = TRUE;
            }
            else
            {
                // Jump to next item in the list
                pEntry = pEntry->Flink;
            }
        }

        KeReleaseSpinLock(
            &(pdx->Lock_PhysicalMemList),
            IrqL_Original
            );

        if (!bFound)
        {
            DebugPrintf(("ERROR - Physical memory object not found in list, unable to unmap\n"));
            return ApiInvalidAddress;
        }
    }

    // Do nothing if no mappings exist
    if (IsListEmpty(&pMemObject->List_Mappings))
        return ApiInvalidAddress;

    // Find the map object to unmap
    KeAcquireSpinLock(
        &(pMemObject->Lock_MappingsList),
        &IrqL_Original
        );

    pEntry = pMemObject->List_Mappings.Flink;

    bFound = FALSE;

    // Traverse list to find the desired list object
    while (!bFound && (pEntry != &(pMemObject->List_Mappings)))
    {
        // Get the object
        pMapObject =
            CONTAINING_RECORD(
                pEntry,
                PLX_USER_MAPPING,
                ListEntry
                );

        // Compare owner & virtual address
        if ((pOwner == pMapObject->pOwner) &&
            ((PLX_UINT_PTR)pMapObject->pUserVa[0] == pPciMem->UserAddr))
        {
            // Remove entry from the list
            RemoveEntryList(
                pEntry
                );

            bFound = TRUE;
        }
        else
        {
            // Jump to next item in the list
            pEntry = pEntry->Flink;
        }
    }

    KeReleaseSpinLock(
        &(pMemObject->Lock_MappingsList),
        IrqL_Original
        );

    if (!bFound)
    {
        DebugPrintf(("ERROR - Map object not found in list, unable to unmap\n"));
        return ApiInvalidAddress;
    }

    // Unmap the memory
    if (pMapObject->pUserVa[0] == NULL)
    {
        DebugPrintf((
            "ERROR - Invalid virtual address (%p), cannot unmap\n",
            pMapObject->pUserVa[0]
            ));

        return ApiInvalidAddress;
    }

    DebugPrintf((
        "Unmap User VA (%p) for physical memory (owner=%p)\n",
        pMapObject->pUserVa[0], pOwner
        ));

    // Unmap memory from user space
    i    = 0;
    pMdl = pMemObject->pMdl;

    while ((i < MAX_VIRTUAL_ADDR) && (pMapObject->pUserVa[i] != NULL))
    {
        MmUnmapLockedPages(
            pMapObject->pUserVa[i],
            pMdl
            );

        i++;
        pMdl = pMdl->Next;
    }

    // Clear virtual address
    pPciMem->UserAddr = 0;

    // Release memory for map object
    ExFreePool( pMapObject );

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxInterruptEnable
 *
 * Description:  Enables specific interupts of the PLX Chip
 *
 ******************************************************************************/
PLX_STATUS
PlxInterruptEnable(
    DEVICE_EXTENSION *pdx,
    PLX_INTERRUPT    *pPlxIntr
    )
{
    U32          RegValue;
    PLX_REG_DATA RegData;


    // Only 16 doorbell interrupts are supported
    pPlxIntr->Doorbell &= 0xFFFF;

    // Enable doorbell interrupts
    if (pPlxIntr->Doorbell)
    {
        PLX_PCI_REG_READ(
            pdx,
            0xc4,
            &RegValue
            );

        RegValue |= pPlxIntr->Doorbell;

        PLX_PCI_REG_WRITE(
            pdx,
            0xc4,
            RegValue
            );
    }

    // Setup to synchronize access to interrupt register
    RegData.pdx         = pdx;
    RegData.offset      = 0xc8;
    RegData.BitsToSet   = 0;
    RegData.BitsToClear = 0;

    if (pPlxIntr->Message & (1 << 0))
        RegData.BitsToSet |= (1 << 24);

    if (pPlxIntr->Message & (1 << 1))
        RegData.BitsToSet |= (1 << 25);

    if (pPlxIntr->Message & (1 << 2))
        RegData.BitsToSet |= (1 << 26);

    if (pPlxIntr->Message & (1 << 3))
        RegData.BitsToSet |= (1 << 27);

    if (pPlxIntr->ResetDeassert)
        RegData.BitsToSet |= (1 << 28);

    if (pPlxIntr->PmeDeassert)
        RegData.BitsToSet |= (1 << 29);

    if (pPlxIntr->GPIO_14_15)
        RegData.BitsToSet |= (1 << 30);

    if (pPlxIntr->GPIO_4_5)
        RegData.BitsToSet |= (1 << 31);

    // Write register values if they have changed
    if (RegData.BitsToSet != 0)
    {
        // Synchronize write to interrupt register
        KeSynchronizeExecution(
            pdx->pInterruptObject,
            PlxSynchronizedRegisterModify,
            &RegData
            );
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxInterruptDisable
 *
 * Description:  Disables specific interrupts of the PLX Chip
 *
 ******************************************************************************/
PLX_STATUS
PlxInterruptDisable(
    DEVICE_EXTENSION *pdx,
    PLX_INTERRUPT    *pPlxIntr
    )
{
    U32          RegValue;
    PLX_REG_DATA RegData;


    // Only 16 doorbell interrupts are supported
    pPlxIntr->Doorbell &= 0xFFFF;

    // Disable doorbell interrupts
    if (pPlxIntr->Doorbell)
    {
        PLX_PCI_REG_READ(
            pdx,
            0xc4,
            &RegValue
            );

        // Clear doorbell interrupts that are set
        RegValue &= 0xFFFF0000 | (~pPlxIntr->Doorbell & 0xFFFF);

        PLX_PCI_REG_WRITE(
            pdx,
            0xc4,
            RegValue
            );
    }

    // Setup to synchronize access to interrupt register
    RegData.pdx         = pdx;
    RegData.offset      = 0xc8;
    RegData.BitsToSet   = 0;
    RegData.BitsToClear = 0;

    if (pPlxIntr->Message & (1 << 0))
        RegData.BitsToClear |= (1 << 24);

    if (pPlxIntr->Message & (1 << 1))
        RegData.BitsToClear |= (1 << 25);

    if (pPlxIntr->Message & (1 << 2))
        RegData.BitsToClear |= (1 << 26);

    if (pPlxIntr->Message & (1 << 3))
        RegData.BitsToClear |= (1 << 27);

    if (pPlxIntr->ResetDeassert)
        RegData.BitsToClear |= (1 << 28);

    if (pPlxIntr->PmeDeassert)
        RegData.BitsToClear |= (1 << 29);

    if (pPlxIntr->GPIO_14_15)
        RegData.BitsToClear |= (1 << 30);

    if (pPlxIntr->GPIO_4_5)
        RegData.BitsToClear |= (1 << 31);

    // Write register values if they have changed
    if (RegData.BitsToClear != 0)
    {
        // Synchronize write to interrupt register
        KeSynchronizeExecution(
            pdx->pInterruptObject,
            PlxSynchronizedRegisterModify,
            &RegData
            );
    }

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxNotificationRegisterFor
 *
 * Description:  Registers a wait object for notification on interrupt(s)
 *
 ******************************************************************************/
PLX_STATUS
PlxNotificationRegisterFor(
    DEVICE_EXTENSION  *pdx,
    PLX_INTERRUPT     *pPlxIntr,
    PLX_NOTIFY_OBJECT *pEvent,
    VOID              *pOwner
    )
{
    PKEVENT          pKEvent;
    NTSTATUS         status;
    PLX_WAIT_OBJECT *pWaitObject;


    // Setup driver access to the user event handle
    status =
        ObReferenceObjectByHandle(
            (HANDLE)pEvent->hEvent, // Handle
            EVENT_MODIFY_STATE,     // Desired Access
            *ExEventObjectType,     // Object type
            KernelMode,             // Access Mode
            &pKEvent,               // Object pointer
            NULL                    // Handle information
            );

    if (status != STATUS_SUCCESS)
        return ApiInvalidHandle;

    // Allocate a new wait object
    pWaitObject =
        ExAllocatePoolWithTag(
            NonPagedPool,
            sizeof(PLX_WAIT_OBJECT),
            '_XLP'        // Tag = "PLX_" in reverse order
            );

    if (pWaitObject == NULL)
    {
        DebugPrintf((
            "ERROR - memory allocation for interrupt wait object failed\n"
            ));

        // De-reference the object
        ObDereferenceObject(
            pKEvent
            );

        return ApiInsufficientResources;
    }

    // Record the wait object
    pEvent->pWaitObject = (PLX_UINT_PTR)pWaitObject;

    // Record the owner
    pWaitObject->pOwner = pOwner;

    // Store kernel event handle
    pWaitObject->pKEvent = pKEvent;

    // Clear interrupt source
    pWaitObject->Source_Ints     = INTR_TYPE_NONE;
    pWaitObject->Source_Doorbell = 0;

    // Set interrupt notification flags
    pWaitObject->Notify_Flags    = INTR_TYPE_NONE;
    pWaitObject->Notify_Doorbell = pPlxIntr->Doorbell;

    if (pPlxIntr->ResetDeassert)
        pWaitObject->Notify_Flags |= INTR_TYPE_RSTIN;

    if (pPlxIntr->PmeDeassert)
        pWaitObject->Notify_Flags |= INTR_TYPE_PME;

    if (pPlxIntr->GPIO_4_5)
        pWaitObject->Notify_Flags |= INTR_TYPE_GPIO_4_5;

    if (pPlxIntr->GPIO_14_15)
        pWaitObject->Notify_Flags |= INTR_TYPE_GPIO_14_15;

    if (pPlxIntr->Message & (1 << 0))
        pWaitObject->Notify_Flags |= INTR_TYPE_MESSAGE_0;

    if (pPlxIntr->Message & (1 << 1))
        pWaitObject->Notify_Flags |= INTR_TYPE_MESSAGE_1;

    if (pPlxIntr->Message & (1 << 2))
        pWaitObject->Notify_Flags |= INTR_TYPE_MESSAGE_2;

    if (pPlxIntr->Message & (1 << 3))
        pWaitObject->Notify_Flags |= INTR_TYPE_MESSAGE_3;

    // Add to list of waiting objects
    ExInterlockedInsertTailList(
        &(pdx->List_WaitObjects),
        &(pWaitObject->ListEntry),
        &(pdx->Lock_WaitObjectsList)
        );

    DebugPrintf((
        "Registered interrupt wait object (%p)\n",
        pWaitObject
        ));

    return ApiSuccess;
}




/*******************************************************************************
 *
 * Function   :  PlxNotificationWait
 *
 * Description:  Put the process to sleep until wake-up event occurs or timeout
 *
 ******************************************************************************/
PLX_STATUS
PlxNotificationWait(
    DEVICE_EXTENSION  *pdx,
    PLX_NOTIFY_OBJECT *pEvent,
    PLX_UINT_PTR       Timeout_ms
    )
{
    // Implemented at API level in Windows driver
    return ApiUnsupportedFunction;
}




/*******************************************************************************
 *
 * Function   :  PlxNotificationStatus
 *
 * Description:  Returns the interrupt(s) that have caused notification events
 *
 ******************************************************************************/
PLX_STATUS
PlxNotificationStatus(
    DEVICE_EXTENSION  *pdx,
    PLX_NOTIFY_OBJECT *pEvent,
    PLX_INTERRUPT     *pPlxIntr
    )
{
    KIRQL               IrqL_Original;
    PLIST_ENTRY         pEntry;
    PLX_WAIT_OBJECT    *pWaitObject;
    PLX_INTERRUPT_DATA  IntData;


    KeAcquireSpinLock(
        &(pdx->Lock_WaitObjectsList),
        &IrqL_Original
        );

    pEntry = pdx->List_WaitObjects.Flink;

    // Traverse list to find the desired list object
    while (pEntry != &(pdx->List_WaitObjects))
    {
        // Get the object
        pWaitObject =
            CONTAINING_RECORD(
                pEntry,
                PLX_WAIT_OBJECT,
                ListEntry
                );

        // Check if desired object
        if ((PLX_UINT_PTR)pWaitObject == pEvent->pWaitObject)
        {
            // Copy the interrupt sources
            IntData.Source_Ints     = pWaitObject->Source_Ints;
            IntData.Source_Doorbell = pWaitObject->Source_Doorbell;

            // Reset interrupt sources
            pWaitObject->Source_Ints     = INTR_TYPE_NONE;
            pWaitObject->Source_Doorbell = 0;

            KeReleaseSpinLock(
                &(pdx->Lock_WaitObjectsList),
                IrqL_Original
                );

            DebugPrintf((
                "Returning status for interrupt wait object (%p)...\n",
                pWaitObject
                ));

            // Set triggered interrupts
            RtlZeroMemory(
                pPlxIntr,
                sizeof(PLX_INTERRUPT)
                );

            pPlxIntr->Doorbell = IntData.Source_Doorbell;

            if (IntData.Source_Ints & INTR_TYPE_RSTIN)
                pPlxIntr->ResetDeassert = 1;

            if (IntData.Source_Ints & INTR_TYPE_PME)
                pPlxIntr->PmeDeassert = 1;

            if (IntData.Source_Ints & INTR_TYPE_GPIO_4_5)
                pPlxIntr->GPIO_4_5 = 1;

            if (IntData.Source_Ints & INTR_TYPE_GPIO_14_15)
                pPlxIntr->GPIO_14_15 = 1;

            if (IntData.Source_Ints & INTR_TYPE_MESSAGE_0)
                pPlxIntr->Message |= (1 << 0);

            if (IntData.Source_Ints & INTR_TYPE_MESSAGE_1)
                pPlxIntr->Message |= (1 << 1);

            if (IntData.Source_Ints & INTR_TYPE_MESSAGE_2)
                pPlxIntr->Message |= (1 << 2);

            if (IntData.Source_Ints & INTR_TYPE_MESSAGE_3)
                pPlxIntr->Message |= (1 << 3);

            return ApiSuccess;
        }

        // Jump to next item in the list
        pEntry = pEntry->Flink;
    }

    KeReleaseSpinLock(
        &(pdx->Lock_WaitObjectsList),
        IrqL_Original
        );

    return ApiFailed;
}




/*******************************************************************************
 *
 * Function   :  PlxNotificationCancel
 *
 * Description:  Cancels a registered notification event
 *
 ******************************************************************************/
PLX_STATUS
PlxNotificationCancel(
    DEVICE_EXTENSION  *pdx,
    PLX_NOTIFY_OBJECT *pEvent,
    VOID              *pOwner
    )
{
    KIRQL            IrqL_Original;
    BOOLEAN          bRemove;
    PLIST_ENTRY      pEntry;
    PLX_WAIT_OBJECT *pWaitObject;


    KeAcquireSpinLock(
        &(pdx->Lock_WaitObjectsList),
        &IrqL_Original
        );

    pEntry = pdx->List_WaitObjects.Flink;

    // Find the object and remove it
    while (pEntry != &(pdx->List_WaitObjects))
    {
        // Get the object
        pWaitObject =
            CONTAINING_RECORD(
                pEntry,
                PLX_WAIT_OBJECT,
                ListEntry
                );

        // Default to not remove
        bRemove = FALSE;

        // Determine if object should be removed
        if (pOwner == pWaitObject->pOwner)
        {
            if (pEvent == NULL)
            {
                bRemove = TRUE;
            }
            else if ((PLX_UINT_PTR)pWaitObject == pEvent->pWaitObject)
            {
                bRemove = TRUE;
            }
        }

        // Remove object
        if (bRemove)
        {
            DebugPrintf((
                "Removing interrupt wait object (%p)...\n",
                pWaitObject
                ));

            // Remove the object from the list
            RemoveEntryList(
                pEntry
                );

            KeReleaseSpinLock(
                &(pdx->Lock_WaitObjectsList),
                IrqL_Original
                );

            // Signal wait object in case thread is waiting for it
            KeSetEvent(
                pWaitObject->pKEvent,
                IO_NO_INCREMENT,
                FALSE
                );

            // De-reference the kernel event object
            ObDereferenceObject(
                pWaitObject->pKEvent
                );

            // Release the list object
            ExFreePool(
                pWaitObject
                );

            // Return if removing only a specific object
            if (pEvent != NULL)
                return ApiSuccess;

            // Reset to beginning of list
            KeAcquireSpinLock(
                &(pdx->Lock_WaitObjectsList),
                &IrqL_Original
                );

            pEntry = pdx->List_WaitObjects.Flink;
        }
        else
        {
            // Jump to next item in the list
            pEntry = pEntry->Flink;
        }
    }

    KeReleaseSpinLock(
        &(pdx->Lock_WaitObjectsList),
        IrqL_Original
        );

    return ApiFailed;
}
