@echo off

rem **************************************************************
rem *
rem * File       :  BuildDriver.bat
rem *
rem * Description:  Used to build PLX device drivers
rem *
rem **************************************************************


rem ====================================================================
rem  Clean environment from any previous failed builds
rem ====================================================================
SET CL=
SET PLX_CHIP=
IF DEFINED PLX_BUILD_ALT_DIR SET BUILD_ALT_DIR=%PLX_BUILD_ALT_DIR%
SET PLX_BUILD_ALT_DIR=


rem ====================================================================
rem  Perform any error checks
rem ====================================================================
IF /I "%1" == "9050" SET PLX_CHIP=9050
IF /I "%1" == "9030" SET PLX_CHIP=9030
IF /I "%1" == "9080" SET PLX_CHIP=9080
IF /I "%1" == "9054" SET PLX_CHIP=9054
IF /I "%1" == "9056" SET PLX_CHIP=9056
IF /I "%1" == "9656" SET PLX_CHIP=9656
IF /I "%1" == "8311" SET PLX_CHIP=8311

rem Flag that a 9000-series device is building
IF DEFINED PLX_CHIP SET b9000=%PLX_CHIP%

IF /I "%1" == "6000n" SET PLX_CHIP=6000_NT
IF /I "%1" == "8000n" SET PLX_CHIP=8000_NT
IF /I "%1" == "8000d" SET PLX_CHIP=8000_DMA
IF /I "%1" == "Svc"   SET PLX_CHIP=Svc
IF /I "%1" == "ND"    SET PLX_CHIP=8000_ND
IF DEFINED PLX_CHIP GOTO EnvOk

echo.
echo PLX Windows driver build batch file
echo Copyright (c) PLX Technology, Inc.
echo.
echo   Usage: builddriver {PLX_Chip} [CleanOption]
echo.
echo      PLX_Chip    = 8000n      : PLX 8000 NT-mode PnP driver
echo                    8000d      : PLX 8000 DMA Controller PnP driver
echo                    9050       : PLX 9050/9052 PnP driver
echo                    9030       : PLX 9030 PnP driver
echo                    9080       : PLX 9080 PnP driver
echo                    9054       : PLX 9054 PnP driver
echo                    9056       : PLX 9056 PnP driver
echo                    9656       : PLX 9656 PnP driver
echo                    8311       : PLX 8311 PnP driver
echo                    6000n      : PLX 6254/6540/6466 NT-mode PnP driver
echo                    Svc        : PLX PCI/PCIe Service driver
echo.
echo      CleanOption = {none}     : Build the driver
echo                    'clean'    : Remove intermediate build files
echo                    'cleanall' : Remove all build files
GOTO Exit


:ErrorSourceDir
echo.
echo ERROR: Source directory "Source.%CMN%" does not exist, unable to continue
GOTO Exit


:EnvOk


rem ====================================================================
rem  Set target architecture
rem ====================================================================
SET ARCH=%_BUILDARCH%
if /I "%_BUILDARCH%" == "x86" SET ARCH=i386


rem ====================================================================
rem  Override object folder for 9000 devices
rem ====================================================================
IF "%b9000%" == "" Goto _SkipObj9000
SET PLX_BUILD_ALT_DIR=%BUILD_ALT_DIR%
SET BUILD_ALT_DIR=_Plx%PLX_CHIP%_%BUILD_ALT_DIR%
:_SkipObj9000


rem ====================================================================
rem  Switch to correct source directory
rem ====================================================================
IF "%b9000%" == "" (SET CMN=Plx%PLX_CHIP%) ELSE (SET CMN=Plx9000)
IF NOT EXIST Source.%CMN% GOTO ErrorSourceDir
CD Source.%CMN%
SET CMN=


rem ====================================================================
rem  Check for command-line parameters
rem ====================================================================
IF /I "%2" == "C"        SET CL=Clean
IF /I "%2" == "clean"    SET CL=Clean
IF /I "%2" == "A"        SET CL=CleanAll
IF /I "%2" == "cleanall" SET CL=CleanAll


rem ====================================================================
rem  Display parameters
rem ====================================================================
IF DEFINED CL goto _SkipCls
IF DEFINED PLX_AUTO_BLD goto _SkipCls
CLS
:_SkipCls
IF "%PLX_LOG_TO_FILE%"=="1" SET PLX_DEBUG_DEST=[Log to file]

echo.
IF     "%PLX_CHIP%" == "Svc" echo  - TYP: PLX PCI/PCIe Service Driver
IF NOT "%PLX_CHIP%" == "Svc" echo  - TYP: WDM Driver
echo  - CPU: %ARCH%
echo  - CFG: %DDKBUILDENV% %PLX_DEBUG_DEST%
echo  - PLX: %PLX_CHIP%
echo.


rem ====================================================================
rem  If clean option, jump to it
rem ====================================================================
IF DEFINED CL goto %CL%


rem ====================================================================
rem  Create necessary directories
rem ====================================================================
IF NOT EXIST Driver_Plx%PLX_CHIP%  Md Driver_Plx%PLX_CHIP%
IF NOT EXIST %ARCH%                Md %ARCH%


rem ====================================================================
rem  Copy chip-specific files for 9000 devices
rem ====================================================================
IF NOT DEFINED b9000 Goto _SkipCopy9000
echo Copy chip-specific files...
IF EXIST Chip\%PLX_CHIP%\*.c  XCopy Chip\%PLX_CHIP%\*.c  %ARCH% /D /Q >NUL
echo.
:_SkipCopy9000


rem ====================================================================
rem  Build driver
rem     /Z - No dependency checking or scanning of source files - 3 passes
rem     /b - Displays full error message text (doesn't truncate)
rem     /w - Show warnings on screen
rem     /e - Generates build.log, build.wrn & build.err files
rem     /g - Display warnings/errors/summary in color
rem     /I - Do not display thread index if multiprocessor build
rem ====================================================================
build /Z /b /w /e /g /I
IF ERRORLEVEL == 1 GOTO Cleanup


rem ====================================================================
rem  Remove log files if error-free build
rem ====================================================================
IF EXIST Build%BUILD_ALT_DIR%.*  Del Build%BUILD_ALT_DIR%.*


rem ====================================================================
rem  Digitally sign the driver if PLX internal build
rem ====================================================================
IF NOT EXIST ..\..\..\PlxDriverSign goto _DriverSign_Skip
rem Driver signing not supported in older WDKs
IF %_NT_TARGET_VERSION:~0,3% == 0x5 goto _DriverSign_Skip


echo      ______________________________________________________________
echo.
echo DIGITAL SIGN: Driver_Plx%PLX_CHIP%\%DDKBUILDENV%\%_BUILDARCH%\Plx%PLX_CHIP%.sys

rem ***************************************************************
rem * Build command-line in environment variable
rem *
rem * /a         : Auto-select best signing method
rem * /v         : Verbose output
rem * /ac <file> : Add cross-certificate (required for kernel signing)
rem * /f <file>  : Signing certificate
rem * /p <pwd>   : Password for PFX file if needed
rem * /n <name>  : Subject name of certificate
rem * /d <desc>  : Description of signed content
rem * /t <URL>   : Timestamp server URL
rem *
rem ***************************************************************
rem Vista & 2008 WDK report /f & /ac as incompatible
IF NOT %_NT_TARGET_VERSION% == 0x600 (
    SET PLX_SIGN=/f ..\..\..\PlxDriverSign\PLX_Tech.pfx /p plxtechsoftware1
)
SET PLX_SIGN=%PLX_SIGN% /a /v /n "PLX Technology, Inc." /d "PLX SDK driver"
SET PLX_SIGN=%PLX_SIGN% /ac ..\..\..\PlxDriverSign\MSCV-GlobalSign.cer
SET PLX_SIGN=%PLX_SIGN% /t http://timestamp.globalsign.com/scripts/timestamp.dll
SET PLX_SIGN=%PLX_SIGN% Driver_Plx%PLX_CHIP%\%DDKBUILDENV%\%ARCH%\Plx%PLX_CHIP%.sys

Signtool sign %PLX_SIGN% > NUL
IF ERRORLEVEL == 0 SET PLX_SIGN=...sign complete
IF ERRORLEVEL == 1 SET PLX_SIGN=ERROR[%ERRORLEVEL%]: Unable to sign driver
IF ERRORLEVEL == 2 SET PLX_SIGN=WARNING[%ERRORLEVEL%]: Unable to sign driver
echo DIGITAL SIGN: %PLX_SIGN%

IF ERRORLEVEL 1 goto _DriverSign_SkipVerify
echo DIGITAL SIGN: Verify kernel signature
Signtool verify /v /kp Driver_Plx%PLX_CHIP%\%DDKBUILDENV%\%ARCH%\Plx%PLX_CHIP%.sys > NUL
IF ERRORLEVEL == 0 SET PLX_SIGN=...verify complete
IF ERRORLEVEL == 1 SET PLX_SIGN=ERROR[%ERRORLEVEL%]: Driver signature not valid
IF ERRORLEVEL == 2 SET PLX_SIGN=WARNING[%ERRORLEVEL%]: Driver signature not valid
echo DIGITAL SIGN: %PLX_SIGN%

:_DriverSign_SkipVerify
IF /I "%PLX_SIGN:~-8%" == "Complete" goto _DriverSign_Ok
echo DIGITAL SIGN: Verify PLX ^& cross certificates (PFX ^& CER) are valid

rem Add error information to log file
echo. >> Build%BUILD_ALT_DIR%.err
echo ============================================== >> Build%BUILD_ALT_DIR%.err
echo  PLX DIGITAL SIGN: %PLX_SIGN%                  >> Build%BUILD_ALT_DIR%.err
echo ============================================== >> Build%BUILD_ALT_DIR%.err
echo. >> Build%BUILD_ALT_DIR%.err

:_DriverSign_Ok
echo      ______________________________________________________________

:_DriverSign_Skip
SET PLX_SIGN=


rem ====================================================================
rem  Build complete, goto cleanup
rem ====================================================================
goto Cleanup


:CleanAll
rem ====================================================================
rem  Delete all build files if cleaning
rem ====================================================================
echo Delete: All build files
IF EXIST Driver_Plx%PLX_CHIP%  Rd /q /s Driver_Plx%PLX_CHIP%
echo Delete: ...finished!
echo.


:Clean
rem ====================================================================
rem  Delete intermediate build files if cleaning
rem ====================================================================
echo Delete: Intermediate build files
IF EXIST Obj                    Rd /q /s Obj
IF EXIST Obj%BUILD_ALT_DIR%     Rd /q /s Obj%BUILD_ALT_DIR%
IF EXIST Build%BUILD_ALT_DIR%.* Del Build%BUILD_ALT_DIR%.*
echo Delete: ...finished!


:Cleanup
rem Remove common files to avoid multiple versions
IF EXIST %ARCH%  Rd /q /s %ARCH%

rem Go back to original directory
CD ..


:Exit
rem ====================================================================
rem  Cleanup environment
rem ====================================================================
IF DEFINED PLX_BUILD_ALT_DIR SET BUILD_ALT_DIR=%PLX_BUILD_ALT_DIR%
SET PLX_BUILD_ALT_DIR=
SET CL=
SET ARCH=
SET b9000=
SET PLX_CHIP=
SET PLX_DEBUG_DEST=
echo.
